import express from 'express';
import path from 'path';
import fs from 'fs/promises';
import logger from './logger.js';
import axios from 'axios';
import cors from 'cors';

import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const app = express();
const port = process.env.PORT || 3001;
const host = process.env.HOST || '0.0.0.0';

// Middleware
app.use(cors());
app.use(express.json());


// Request logging middleware
app.use((req, res, next) => {
  const start = Date.now();
  res.on('finish', () => {
    const duration = Date.now() - start;
    logger.info(
       `[${req.method}] ${req.path} ${res.statusCode} ${duration}ms ${req.ip}`
    );
  });
  next();
});

const API_KEY_PATH = path.resolve('apiKey.json');
const WHOIS_API_URL = 'https://reverse-whois.whoisxmlapi.com/api/v2';
const BULK_WHOIS_URL = 'https://api.whoisfreaks.com/v1.0/bulkwhois';

// Helper to read API key
const getApiKey = async () => {
  try {
    const data = await fs.readFile(API_KEY_PATH, 'utf8');
    return JSON.parse(data);
  } catch (error) {
    logger.error('Error reading API key', { message: error.message });
    return null;
  }
};

// Proxy route for domain search
app.post('/api/v2', async (req, res) => {
  try {
    const { whoisXMLapiKey: apiKey } = await getApiKey();
    if (!apiKey) {
      return res.status(500).json({ error: 'API key not configured.' });
    }

    const response = await axios.post(WHOIS_API_URL, {
      ...req.body,
      apiKey,
    });
    const { domainsCount, domainsList } = response.data;
    return res.status(200).json({ count: req.body?.limit || domainsCount, list: domainsList.slice(0, req.body.limit || 100) });
  } catch (error) {


    if (error?.response?.status === 401 || error?.response?.status === 403) {
      return res.status(400).json({ error: 'Your WhoisXMLApi  key is invalid or expired. Please update it in Settings.' })
     
    }
    logger.error('Proxy error', {
      message: error.response ? error.response.data : error.message,
      path: req.path,
      ip: req.ip,
      requestBody: req.body,
      status: error.response?.status || 500,
    });
    res.status(error.response?.status || 500).json({ error: 'Failed to fetch data from Whois API.' });
  }
});

// Proxy route for bulk domain details
const formatField = (value) => {
  if (!value) {
    return "N/A";
  }
  const lowerCaseValue = String(value).toLowerCase();
  if (

    lowerCaseValue.includes("REDACTED FOR PRIVACY".toLowerCase()) || lowerCaseValue.includes("Please query the RDDS service".toLowerCase()) || lowerCaseValue.includes("REDACTEDFORPRIVACY".toLowerCase()) || lowerCaseValue.includes("REDACTED".toLowerCase())

  ) {
    return "Redacted for privacy";
  }
  return value;
};

const formatDate = (dateString) => {
  // If the date string is null, empty, or 'N/A', return 'N/A'
  if (!dateString || dateString === 'N/A') {
    return 'N/A';
  }

  try {
    // Create a new Date object from the input string.
    const date = new Date(dateString);

    // Check if the date is valid.
    if (isNaN(date.getTime())) {
      return 'N/A';
    }

    // Convert the date to an ISO string (e.g., "2024-09-14T10:30:00.000Z")
    // and slice the first 10 characters to get the "YYYY-MM-DD" part.
    return date.toISOString().slice(0, 10);
  } catch (error) {
    // If any error occurs during parsing, return 'N/A'.
    return 'N/A';
  }
};
app.post('/api/bulk-whois', async (req, res) => {
  try {
    const { whoisFreakapiKey: apiKey } = await getApiKey();

    if (!apiKey) {
      return res.status(500).json({ error: 'API key not configured.' });
    }

    const response = await axios.post(`${BULK_WHOIS_URL}?apiKey=${apiKey}`, {
      domainNames: req.body.domains,
    });

    if (response.status === 400) {
      return res.status(400).json({ message: 'Invalid request.' });
    }

    const { bulk_whois_response } = response.data;

    return res.status(200).json({
      count: bulk_whois_response.length,
      list: bulk_whois_response.map((r) => ({
        domain: r.domain_name,
        creationDate: formatDate(r?.create_date || 'N/A'),
        email: formatField(r?.registrant_contact?.email_address),
        name: formatField(r.registrant_contact?.name),
        phone: formatField(r?.registrant_contact?.phone),
        country: formatField(r?.registrant_contact?.country_name),
        city: formatField(r?.registrant_contact?.city),
      })),
    });
  } catch (error) {
    if (error.response && error.response.status === 401) {
      logger.error('WhoisFreak API Error. Switching to WhoisXML API.', 
        {
          message: error.response ? error.response.data : error.message,
          path: req.path,
          ip: req.ip,
          requestBody: req.body,
          status: error.response?.status || 500,
        }
      );

      try {
        const { whoisXMLapiKey } = await getApiKey();
        const domains = req.body.domains;

        // Step 1: Send domains and get requestId
        const sendDomainsResponse = await axios.post(
          'https://www.whoisxmlapi.com/BulkWhoisLookup/bulkServices/bulkWhois',
          {
            apiKey: whoisXMLapiKey,
            domains: domains,
            outputFormat: 'JSON',
          }
        );

        const { requestId } = sendDomainsResponse.data;
 

        if (!requestId) {
          throw new Error('Failed to get requestId from WhoisXML API.');
        }

    

        // Polling logic to wait for records to be processed
        let getRecordsResponse;
        let attempts = 0;
        const maxAttempts = 10;
        const pollInterval = 15*1000; // 15 seconds

        while (attempts < maxAttempts) {
          getRecordsResponse = await axios.post(
            'https://www.whoisxmlapi.com/BulkWhoisLookup/bulkServices/getRecords',
            {
              apiKey: whoisXMLapiKey,
              requestId: requestId,
              outputFormat: 'JSON',
              maxRecords: domains.length,
              startIndex: 1,
            }
          );

          const { recordsLeft } = getRecordsResponse.data;
          

          if (recordsLeft === 0) {
            break; // All records processed, and we have them in getRecordsResponse
          }

          attempts++;
          if (attempts >= maxAttempts) {
            throw new Error('Polling timed out waiting for WhoisXML API to process records.');
          }

          await new Promise(resolve => setTimeout(resolve, pollInterval));
        }

        const { whoisRecords } = getRecordsResponse.data;
        return res.status(200).json({
          count: whoisRecords.length,
          list: whoisRecords.map((r) => ({
            domain: r.domainName,
            creationDate: formatDate(r.whoisRecord?.createdDate || r.whoisRecord?.registryData?.createdDate),
            email: formatField(r.whoisRecord?.registrant?.email|| r.whoisRecord?.registryData?.registrant?.email),
            name: formatField(r.whoisRecord?.registrant?.name || r.whoisRecord?.registryData?.registrant?.name),
            phone: formatField(r.whoisRecord?.registrant?.telephone || r.whoisRecord?.registryData?.registrant?.telephone),
            country: formatField(r.whoisRecord?.registrant?.country || r.whoisRecord?.registryData?.registrant?.country),
            city: formatField(r.whoisRecord?.registrant?.city || r.whoisRecord?.registryData?.registrant?.city),
          })),
        });
      } catch (xmlError) {
        logger.error('WhoisXML API request failed', {
          message: xmlError.response ? xmlError.response.data : xmlError.message,
          path: req.path,
          ip: req.ip,
          requestBody: req.body,
          status: xmlError.response?.status || 500,
        });
        return res.status(500).json({ error: 'Failed to fetch data from WhoisXML API.' });
      }
    }

    logger.error('Bulk Whois Proxy error', {
      message: error.response ? error.response.data : error.message,
      path: req.path,
      ip: req.ip,
      requestBody: req.body,
      status: error.response?.status || 500,
      error: error?.response?.data || error.message,
    });
    res.status(error.response?.status || 500).json({ error: 'Failed to fetch data from Bulk Whois API.' });
  }
});


// Route to update API keys
app.get('/api/logs', async (req, res) => {
  try {
    const logFilePath = path.join(__dirname, 'error.log');
    const data = await fs.readFile(logFilePath, 'utf8');
    res.set('Content-Type', 'text/plain');
    res.send(data);
  } catch (err) {
    logger.error('Failed to read error log file', { message: err.message });
    res.status(500).send('Failed to read log file.');
  }
});

app.put('/api/keys', async (req, res) => {
  const { whoisXMLapiKey, whoisFreakapiKey } = req.body;

  if (!whoisXMLapiKey && !whoisFreakapiKey) {
    return res.status(400).json({ error: 'At least one API key is required.' });
  }

  try {
    const currentKeys = await getApiKey() || {};
    const updatedKeys = { ...currentKeys };

    if (whoisXMLapiKey) {
      updatedKeys.whoisXMLapiKey = whoisXMLapiKey;
    }

    if (whoisFreakapiKey) {
      updatedKeys.whoisFreakapiKey = whoisFreakapiKey;
    }

    await fs.writeFile(API_KEY_PATH, JSON.stringify(updatedKeys, null, 2));
    res.status(200).json({ message: 'API keys updated successfully.' });
  } catch (error) {
    logger.error('Error updating API keys', {
      message: error.message,
      path: req.path,
      ip: req.ip,
      requestBody: req.body,
      status: error.response?.status || 500,
    });
    res.status(500).json({ error: 'Failed to update API keys.' });
  }
});

// Route to get API key balance
const maskKey = (key) => {
  if (!key || key.length <= 8) {
    return '****';
  }
  return `${key.substring(0, 4)}...${key.substring(key.length - 4)}`;
};

// Route to get masked API keys
app.get('/api/keys', async (req, res) => {
  try {
    const keys = await getApiKey();
    if (!keys) {
      return res.status(500).json({ error: 'API keys not configured.' });
    }

    res.status(200).json({
      whoisXMLapiKey: keys.whoisXMLapiKey ? maskKey(keys.whoisXMLapiKey) : '',
      whoisFreakapiKey: keys.whoisFreakapiKey ? maskKey(keys.whoisFreakapiKey) : '',
    });
  } catch (error) {
    logger.error('Error fetching keys', { message: error.message, path: req.path, ip: req.ip });
    res.status(500).json({ error: 'Failed to fetch keys.' });
  }
});

// Route to get API key balance
app.get('/api/balance', async (req, res) => {
  try {
    const { whoisXMLapiKey, whoisFreakapiKey } = await getApiKey() || {};

    const promises = [
      whoisXMLapiKey 
        ? axios.get(`https://user.whoisxmlapi.com/user-service/account-balance?apiKey=${whoisXMLapiKey}`)
        : Promise.reject('WhoisXMLAPI key not configured.'),
      whoisFreakapiKey 
        ? axios.get(`https://api.whoisfreaks.com/v1.0/whoisapi/usage?apiKey=${whoisFreakapiKey}`)
        : Promise.reject('WhoisFreakAPI key not configured.')
    ];

    const results = await Promise.allSettled(promises);

    let whoisXMLapiBalance = 'unavailable';
    if (results[0].status === 'fulfilled') {
      const responseData = results[0].value.data;
      whoisXMLapiBalance = responseData?.data?.find(item => item.product_id === 14)?.credits ?? 'unavailable';
    } else {
      logger.error('Error fetching WhoisXMLAPI balance', {
        message: results[0].reason.response ? results[0].reason.response.data : results[0].reason.message || results[0].reason,
        path: req.path,
        ip: req.ip,
      });
    }

    let whoisFreakBalance = 'unavailable';
    if (results[1].status === 'fulfilled') {
      const responseData = results[1].value.data;
      const calculatedBalance = responseData.apiCredits?.totalCredits - responseData.apiCredits?.servedRequest;
      whoisFreakBalance = isNaN(calculatedBalance) ? 'unavailable' : calculatedBalance;
    } else {
      logger.error('Error fetching WhoisFreak balance', {
        message: results[1].reason.response ? results[1].reason.response.data : results[1].reason.message || results[1].reason,
        path: req.path,
        ip: req.ip,
      });
    }

    res.status(200).json({
      whoisXMLapi: whoisXMLapiBalance,
      whoisFreak: whoisFreakBalance,
    });

  } catch (error) {
    logger.error('Error in balance route', {
      message: error.message,
      path: req.path,
      ip: req.ip,
    });
    res.status(500).json({ error: 'An unexpected error occurred while fetching balance.' });
  }
});

app.listen(port, host, () => {
  console.log(`Backend server listening at http://${host}:${port}`);
});
